#!/usr/bin/python
# -*- coding: ISO-8859-15 -*-
import traceback
import sys
import shutil
import zipfile
import os
import magic
import logging

import _common
import common
import db_common
import qry
import fs
import libjobdext
import wf
import docpak

import argparse

script_name = os.path.basename(__file__)
logger = logging.getLogger(script_name)

def extract_zip_and_store_content(gses, parameters):
	jobd_dir = _common.dopath(gses.common.context["dirs.spooldir"], "jobd")
	jobdir = _common.dopath(jobd_dir, parameters["jobqueueid"])
	_common.makealldir(jobdir)

	since_date = parameters["sincedate"].strip()
	if len(since_date) == 10 :
		since_date += " 00:00:00"

	zip_fstypid = db_common.dbgenericgetvalue(usession = gses, itbl = "_fstyp", ifldkey = "FSTYP_EXT",
											  ifldkeyval = "zip", ifldtoreturn = "FSTYP_ID")

	logger.info("Looking for Facture_Pivot zip files since %s" % since_date)

	# Dans ce bloc on va crire toute la logique de notre script.
	myquery = qry.qry(gses)
	myquery.table = [["@ljoin", "_fsfile.FSFILE_ID", "_docpak.DOCPAK_FSFILEID"]]
	myquery.field = [["_fsfile", "FSFILE_ID"], ["_fsfile", "FSFILE_ONAME"], ["_docpak", "DOCPAK_TBL"],
					 ["_docpak", "DOCPAK_RSID"]]

	c1 = [["@fld", "_fsfile", "FSFILE_ONAME"], ["@op", "="], ["@val", "Facture_Pivot.xml"]]
	c2 = [["@fld", "_fsfile", "FSFILE_NSTAMP"], ["@op", ">="], ["@val", since_date]]
	c3 = [["@fld", "_docpak", "DOCPAK_ID"], ["@op", "is not"], ["@null"]]
	myquery.cond = [c1, ["@op", "AND"], [c2, ["@op", "AND"], c3]]

	myquery.order = [["@fld", "_fsfile", "FSFILE_ID", "A"]]

	sql_query = myquery.sql()

	result = gses.db.query(sql_query)
	if result is not None :
		while 1 :
			vrow = gses.db.fetchrow(result)
			if vrow is None :
				break

			fsfileid = vrow["FSFILE_ID"]
			oname = vrow["FSFILE_ONAME"]
			table = vrow["DOCPAK_TBL"]
			rsid = vrow["DOCPAK_RSID"]

			logger.info("\n\tFound file #%d (Table: %s, RSID: %d)" % (fsfileid, table, rsid))

			fileinfos = fs.fsfileloc(
				usession = gses,
				ifileid = fsfileid,
				multiloc = 0
			)

			if fileinfos is None :
				logger.info("\tUnable to get informations about the file. Continue.")
				continue

			zip_path = fileinfos["path"]

			if not os.path.exists(zip_path) :
				logger.info("\t%s does not exists. Continue." % zip_path)

			fp = open(zip_path, "rb")
			zip_content = fp.read()
			fp.close()

			if not magic.guessMime(zip_content) == "application/zip" :
				logger.info("\tNot a zip file. Continue")
				continue
			else :
				logger.info("\tIs a zip file. Process it.")

			# Create extraction directory for the zip
			zip_extraction_dir = _common.dopath(jobdir, str(fsfileid))
			_common.makealldir(zip_extraction_dir)

			# Read the zip and list its content
			chorus_zip = zipfile.ZipFile(zip_path)
			archives_list = chorus_zip.filelist

			if len(archives_list) == 0 :
				logger.info("\tZip is empty. Continue.")
				continue

			# Create storage job
			params = ["tfqn", "rsid", "start", "ocr", "filepath"]
			paramvalues = [table, rsid, -2, 0, "[[files.lst]]"]

			storage_job_id, errormsg = libjobdext.jobcreatefromtpl(
				usession = gses,
				id = "FSSTO",
				secusrid = parameters["secusrid"],
				params = params,
				paramvalues = paramvalues,
				active = False
			)

			if storage_job_id <= 0 :
				logger.error("\tCould not create storage job. Continue.")
				continue

			logger.info("\tStorage job created (#%d)" % storage_job_id)
			storage_job_dir = os.path.join(jobd_dir, str(storage_job_id))
			_common.makealldir(storage_job_dir)

			storage_job_filelist = open(os.path.join(storage_job_dir, "files.lst"), "wb")

			for archive in archives_list :
				archive_filename = archive.filename
				archive_basename = os.path.basename(archive_filename)

				archive_content = chorus_zip.read(archive_filename)

				archive_extraction_path = os.path.join(zip_extraction_dir, archive_basename)

				logger.info("\t\tExtract %s to %s" % (archive_basename, archive_extraction_path))

				fp = open(archive_extraction_path, "wb")
				fp.write(archive_content)
				fp.close()

				if not os.path.exists(archive_extraction_path) :
					logger.error("Error when extracting %s. Continue." % archive_basename)

				archive_dst_path = os.path.join(storage_job_dir, archive_basename)
				logger.info("\t\tCopy %s to %s" % (archive_extraction_path, archive_dst_path))
				shutil.copy(archive_extraction_path, archive_dst_path)

				storage_job_filelist.write(archive_dst_path.encode("iso-8859-15") + "\n")

			logger.info("\tWriting files.lst")
			storage_job_filelist.close()

			logger.info("\tActivate job #%d" % (storage_job_id))
			db_common.dbgenericupdate(
				usession = gses,
				itbl = "_jobqueue",
				iflds = ["JOBQUEUE_ACTIVE"],
				ivalues = [1],
				iidval = storage_job_id
			)

			logger.info("\tRename file #%d from %s to %s" % (fsfileid, oname, oname.replace(".xml", ".zip")))
			db_common.dbgenericupdate(
				usession = gses,
				itbl = "_fsfile",
				iflds = ["FSFILE_ONAME", "FSFILE_FSTYPID"],
				ivalues = [oname.replace(".xml", ".zip"), zip_fstypid],
				iidval = fsfileid
			)

		gses.db.delcursor(result)

def add_extracted_files_to_wfbag(gses, parameters):

	since_date = parameters["sincedate"].strip()
	if len(since_date) == 10 :
		since_date += " 00:00:00"

	zip_fstypid = db_common.dbgenericgetvalue(usession = gses, itbl = "_fstyp", ifldkey = "FSTYP_EXT",
											  ifldkeyval = "zip", ifldtoreturn = "FSTYP_ID")

	logger.info("\nLooking for Facture_Pivot.zip files sended to workflow since %s" % since_date)

	# Dans ce bloc on va crire toute la logique de notre script.
	myquery = qry.qry(gses)
	myquery.table = [["@ljoin", "_fsfile.FSFILE_ID", "_docpak.DOCPAK_FSFILEID"]]
	myquery.field = [["_fsfile", "FSFILE_ID"], ["_fsfile", "FSFILE_ONAME"], ["_docpak", "DOCPAK_ID"],
					 ["_docpak", "DOCPAK_TBL"], ["_docpak", "DOCPAK_RSID"]]

	c1 = [["@fld", "_fsfile", "FSFILE_ONAME"], ["@op", "="], ["@val", "Facture_Pivot.zip"]]
	c2 = [["@fld", "_fsfile", "FSFILE_NSTAMP"], ["@op", ">="], ["@val", since_date]]
	c3 = [["@fld", "_docpak", "DOCPAK_ID"], ["@op", "is not"], ["@null"]]
	myquery.cond = [c1, ["@op", "AND"], [c2, ["@op", "AND"], c3]]

	myquery.order = [["@fld", "_fsfile", "FSFILE_ID", "A"]]

	sql_query = myquery.sql()

	result = gses.db.query(sql_query)
	if result is not None :
		while 1 :
			vrow = gses.db.fetchrow(result)
			if vrow is None :
				break

			docpakid = vrow["DOCPAK_ID"]
			fsfileid = vrow["FSFILE_ID"]
			table    = vrow["DOCPAK_TBL"]
			rsid	 = vrow["DOCPAK_RSID"]

			logger.info("\n\tFound file #%d (Table: %s, RSID: %d)" % (fsfileid, table, rsid))

			wfbagpak = db_common.dbgenericgetrow(
				usession = gses,
				itbl = "_wfbagpak",
				iidval = None,
				icond = [["@fld","_wfbagpak","WFBAGPAK_DOCPAKID"],["@op","="],["@val",docpakid]]
			)


			if wfbagpak is None:
				logger.info("\tNot attached to a workflow bag. Continue.")
				continue

			wfbagid = wfbagpak["WFBAGPAK_WFBAGID"]
			qryid   = wfbagpak["WFBAGPAK_QRYID"]

			logger.info("\tAttached to the workflow bag #%d" % (wfbagid))

			files = docpak.loadallfilesfortblandrsid(usession = gses, maintable = table, rsid = rsid)

			for archive in files:
				fsfileid = archive["DOCPAK_FSFILEID"]
				docpakid = archive["DOCPAK_ID"]

				if not db_common.dbgenericrecordexists(
					usession = gses,
					itbl = "_wfbagpak",
					fields  = ["WFBAGPAK_WFBAGID", "WFBAGPAK_DOCPAKID", "WFBAGPAK_QRYID"],
					values  = [wfbagid, docpakid, qryid]
				):
					logger.info("\tSending file %s (%d) to the workflow on same bag #%d" % (archive["oname"], fsfileid, wfbagid))

					request = _common.FakeRequest()
					request.args["qryid"] = [qryid]
					request.args["wfbagid"] = [wfbagid]
					request.args["docpakid"] = [docpakid]

					wfbagpak = wf.wfbagpak(None)
					wfbagpak.addpages(request, gses)

def execscript(params = None, edb = None) :
	"""
	La fonction execscript est la fonction principale (comme une fonction main en C par exemple)
	qui sera appele par le serveur des travaux afin d'excuter le script.

	@param params: la liste des arguments fournis au script.
	@type params: list

	@param edb: Une instance de connexion  la base de donnes.
	@type edb: instance
	"""
	parameters = {}

	if params is None :
		# Dans le cas d'un appel en ligne de commande on rcupre les arguments qui ont t fournies via cette ligne de commande.
		params = sys.argv

	# Ci-dessous on rcupre dans un dictionnaire python les arguments passs au script.
	ret = _common.getscriptparameters(parameters, params, "instance", "jobqueueid", "secusrid","sincedate")

	if ret <> 0 :
		return ret, str(ret)

	# On	initialisation le contexte et surtout la session ezged.
	gctx = common.common(parameters["instance"])
	gses = common.session(parameters["secusrid"], gctx.context[gctx.context["applic"] + ".default_language"], gctx,
						  parameters["jobqueueid"])

	# On initialise la connexion  la base de donnes via la session ezged.
	if edb == None :
		db_common.sql_dbcontext(gses)
		db = gses.db
	else :
		gses.db = edb
		db = edb

	if db == None :
		return -1, gses.lng("Unable to connect to database")

	else :


		extract_zip_and_store_content(gses, parameters)

		add_extracted_files_to_wfbag(gses, parameters)


	if edb is None :
		# Le script a t appel sans lui fournir une connexion  la bdd.
		# Dans ce cas on ferme la connexion que nous avons ouverte au sein du script.
		db.disconnect()

	return 0, "ok"


if __name__ == "__main__" :
	logger.addHandler(logging.StreamHandler())
	logger.setLevel(logging.DEBUG)

	try :
		errorcode, errorstr = execscript()
	except :
		print "@error"
		traceback.print_exc(file = sys.stdout)
		sys.exit(-100)

	if errorcode <> 0 :
		print "@error", errorstr
	sys.exit(errorcode)

